<?php
namespace App\Http\Controllers;
use Illuminate\Http\Request;
use App\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Twilio\Rest\Client;
use App\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Crypt;
use App\Affilate;
use Mail;
use Session;
use App\Mail\verifyEmail;
use Illuminate\Support\Str;
use App\Mail\WelcomeUser;
use Carbon\Carbon;
use Notification;
use App\Menu;
use App\Notifications\MyNotification;
use App\PaypalSubscription;
use App\Multiplescreen;
use Illuminate\Support\Facades\DB;
use Stripe\Stripe;
use Stripe\Customer;
use App\Button;
use App\Package;

class CustomLoginController extends Controller
{
    private $otpDir;
    private $config;
    private $button;

    public function __construct()
    {
        $this->middleware('guest');
        $this->config = Config::first();
        $this->button = Button::first();
        $this->otpDir = public_path('otp');
        if (!File::exists($this->otpDir)) {
            File::makeDirectory($this->otpDir, 0755, true);
        }
    }

    private function isTwilioEnabled()
    {
        return $this->config->otp_enable ?? false;
    }

    private function generateOTP()
    {
        return rand(100000, 999999);
    }

    private function storeOTPAndData($phone, $otp, $data = null, $isRegistration = false)
    {
        $filename = $this->otpDir . '/' . $phone . '.txt';
        $content = $otp . "\n";
        if ($data) {
            if ($isRegistration) {
                // Encrypt data for registration
                $content .= Crypt::encrypt(json_encode($data)) . "\n";
            } else {
                // Store data without encryption for login
                $content .= json_encode($data) . "\n";
            }
        } else {
            $content .= "\n"; // Empty line if no data
        }
        $expirationTime = time() + 300; // 5 minutes from now
        $content .= $expirationTime;
        File::put($filename, $content);
        $this->startBackgroundDeletion($filename, $expirationTime);
    }

    private function startBackgroundDeletion($filename, $expirationTime)
    {
        $command = "php -r \"sleep(" . ($expirationTime - time()) . "); " .
                   "if (time() >= $expirationTime && file_exists('$filename')) { " .
                   "unlink('$filename'); }\" > /dev/null 2>&1 & echo $!";
        exec($command);
    }

    private function getOTPAndData($phone)
    {
        $filename = $this->otpDir . '/' . $phone . '.txt';
        if (File::exists($filename)) {
            $content = File::get($filename);
            $lines = explode("\n", trim($content));
            $otp = $lines[0]; // First line is the OTP
            $data = null;
            $expirationTime = null;
            if (count($lines) > 1 && !empty($lines[1])) {
                try {
                    // Attempt to decrypt (for registration data)
                    $data = json_decode(Crypt::decrypt($lines[1]), true);
                } catch (\Illuminate\Contracts\Encryption\DecryptException $e) {
                    // If decryption fails, assume it's unencrypted login data
                    $data = json_decode($lines[1], true);
                }
            }
            if (count($lines) > 2) {
                $expirationTime = (int)$lines[2]; // Third line is expiration timestamp
            }
            // Check if OTP has expired
            if ($expirationTime && time() > $expirationTime) {
                return null;
            }
            return [
                'otp' => $otp,
                'data' => $data
            ];
        }
        return null;
    }

    private function deleteOTPFile($phone)
    {
        $filename = $this->otpDir . '/' . $phone . '.txt';
        if (File::exists($filename)) {
            File::delete($filename);
        }
    }

    public function showOTPVerificationForm()
    {
        $phone = session('phone');
        if (!$phone) {
            return redirect()->route('login');
        }
        return view('auth.otp_verification', compact('phone'));
    }

    public function showLoginForm()
    {
        $twilioEnabled = $this->isTwilioEnabled();
        return view('auth.login', compact('twilioEnabled'));
    }

    public function showRegisterForm()
    {
        return view('auth.register');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'email',
            'password' => 'required',
        ]);

        if (Auth::attempt($credentials)) {
            $request->session()->regenerate();
            
            // Set user session data
            session()->put('user', ['email' => Auth::user()->email, 'name' => Auth::user()->name]);
            
            // Use afterLoginProcess method for consistent behavior
            return $this->afterLoginProcess(Auth::user());
        }

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ]);
    }

    public function initiateRegistration(Request $request)
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',
            'full_mobile' => 'required|string|unique:users,mobile',
        ];

        if ($this->config->captcha == 1) {
            $rules['g-recaptcha-response'] = 'required|captcha';
        }

        $request->validate($rules);

        if ($this->isTwilioEnabled()) {
            $otp = $this->generateOTP();
            try {
                $accountSid = env('TWILIO_SID');
                $authToken = env('TWILIO_AUTH_TOKEN');
                $twilioNumber = env('TWILIO_NUMBER');
                $client = new Client($accountSid, $authToken);
                $client->messages->create(
                    $request->full_mobile,
                    [
                        'from' => $twilioNumber,
                        'body' => "Your OTP for registration is: $otp"
                    ]
                );

                $this->storeOTPAndData($request->full_mobile, $otp, $request->only(['name', 'email', 'password', 'full_mobile']), true);
                return redirect()->route('verify.otp.form')->with('phone', $request->full_mobile);
            } catch (\Exception $e) {
                return back()->withErrors(['error' => 'Failed to send OTP: ' . $e->getMessage()]);
            }
        } else {
            $af_system = Affilate::first();
            $user = User::create([
                  'name' => $request['name'],
                  'email' => $request['email'],
                  'mobile' => $request['mobile'],
                  'is_admin' => 0,
                  'status' => 1,
                  'password' => bcrypt($request['password']),
                  'verifyToken' => Str::random(40),
                  'refered_from' => $af_system && $af_system->enable_affilate == '1'  ? $request['refer_code'] : NULL
              ]);
            $user->assignRole('User');
            
            if($request->refer_code != NULL){
              $this->refer($request,$user);
            }
            
            if($this->config->verify_email == 1){
                $user->status = 0;
                $user->save();
                try {
                  Mail::to($user['email'])->send(new verifyEmail($user));
                  return redirect()->route('login')->with('success', __('Verification Email has been sent to your email'));
                } catch (\Exception $e) {
                  \Log::error('Error sending verification email: ' . $e->getMessage());
                  return redirect()->route('login')->with('error', __('Failed to send verification email. Please try again later.'));
                }
            }
            elseif($this->config->wel_eml == 1 && $this->config->verify_email == 0 ){
               if($this->config->free_sub == 1){
                    $this->freesubscribe($user);
                    Auth::login($user);
                    
                    // Set user session data
                    session()->put('user', ['email' => $user->email, 'name' => $user->name]);
                    
                    // Use afterLoginProcess for consistent behavior
                    return $this->afterLoginProcess($user);
                }
                
                try {
                    Mail::to($request['email'])->send(new WelcomeUser($user));
                } catch (\Exception $e) {
                    \Log::error('Error sending welcome email: ' . $e->getMessage());
                }
                
                Auth::login($user);
                
                // Set user session data
                session()->put('user', ['email' => $user->email, 'name' => $user->name]);
                
                // Use afterLoginProcess for consistent behavior
                return $this->afterLoginProcess($user);
            }
            elseif($this->config->wel_eml == 0 && $this->config->verify_email == 0){
                if($this->config->free_sub == 1){
                    $this->freesubscribe($user);
                }
                
                Auth::login($user);
                
                // Set user session data
                session()->put('user', ['email' => $user->email, 'name' => $user->name]);
                
                // Use afterLoginProcess for consistent behavior
                return $this->afterLoginProcess($user);
            }else{
                Auth::login($user);
                
                // Set user session data
                session()->put('user', ['email' => $user->email, 'name' => $user->name]);
                
                // Use afterLoginProcess for consistent behavior
                return $this->afterLoginProcess($user);
            }
        }
    }

    public function resendOTP(Request $request)
    {
        $phone = $request->input('phone');
        if (!$phone) {
            return response()->json(['error' => 'No phone number found for OTP resend.'], 400);
        }

        $storedData = $this->getOTPAndData($phone);
        $newOtp = $this->generateOTP();

        try {
            $accountSid = env('TWILIO_SID');
            $authToken = env('TWILIO_AUTH_TOKEN');
            $twilioNumber = env('TWILIO_NUMBER');
            $client = new Client($accountSid, $authToken);
            $client->messages->create(
                $phone,
                [
                    'from' => $twilioNumber,
                    'body' => "Your new OTP is: $newOtp"
                ]
            );

            $isRegistration = $storedData && isset($storedData['data']['name']);
            $this->storeOTPAndData($phone, $newOtp, $storedData ? $storedData['data'] : null, $isRegistration);
            return response()->json(['success' => 'New OTP sent.']);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to resend OTP: ' . $e->getMessage()], 500);
        }
    }

    public function verifyOTP(Request $request)
    {
        $request->validate([
            'otp' => 'required|numeric',
            'phone' => 'required|string',
        ]);

        $storedData = $this->getOTPAndData($request->phone);
        if (!$storedData) {
            return back()->withErrors(['otp' => 'OTP has expired or is invalid']);
        }

        if ($storedData['otp'] !== $request->otp) {
            return back()->withErrors(['otp' => 'Invalid OTP']);
        }

        // OTP is valid, proceed with registration or login
        if ($storedData['data'] && isset($storedData['data']['name'])) {
            $userData = $storedData['data'];
            // Register new user
            $user = User::create([
                'name' => $userData['name'],
                'email' => $userData['email'],
                'password' => Hash::make($userData['password']),
                'mobile' => $userData['full_mobile'],
                'status' => 1,
                'is_admin' => 0, // Set default value for is_admin
                'email_verified_at' => now(), // Set email as verified
            ]);
            
            $this->deleteOTPFile($request->phone);
            Auth::login($user);
            
            // Set user session data
            session()->put('user', ['email' => $user->email, 'name' => $user->name]);
            
            // Use afterLoginProcess for consistent behavior
            return $this->afterLoginProcess($user);
        } else {
            $user = User::where('mobile', $request->phone)->first();
            if (!$user) {
                return back()->withErrors(['error' => 'User not found']);
            }
            
            Auth::login($user);
            $this->deleteOTPFile($request->phone);
            
            // Set user session data
            session()->put('user', ['email' => $user->email, 'name' => $user->name]);
            
            // Use afterLoginProcess for consistent behavior
            return $this->afterLoginProcess($user);
        }
    }

    public function initiateOTPLogin(Request $request)
    {
        $request->validate([
            'full_mobile' => 'required|string',
        ]);

        $user = User::where('mobile', $request->full_mobile)->first();
        if (!$user) {
            return back()->withErrors(['error' => 'No account found with this phone number.']);
        }

        $otp = $this->generateOTP();
        try {
            $accountSid = env('TWILIO_SID');
            $authToken = env('TWILIO_AUTH_TOKEN');
            $twilioNumber = env('TWILIO_NUMBER');
            $client = new Client($accountSid, $authToken);
            $client->messages->create(
                $request->full_mobile,
                [
                    'from' => $twilioNumber,
                    'body' => "Your OTP for login is: $otp"
                ]
            );

            $this->storeOTPAndData($request->full_mobile, $otp, null, false);
            return redirect()->route('verify.otp.form')->with('phone', $request->full_mobile);
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to send OTP: ' . $e->getMessage()]);
        }
    }

    public function sendEmailDone($email, $verifyToken){
        $user = User::where(['email' => $email, 'verifyToken' => $verifyToken])->first();
        if($user){
            User::where(['email' => $email, 'verifyToken' => $verifyToken])->update(['status'=>'1','verifyToken'=>NULL]);
            Session::flash('success', __('Email Verification Successfull'));
            Mail::to($user->email)->send(new WelcomeUser($user));
            
            if($this->config->free_sub == 1){
                $this->freesubscribe($user);
            }
            
            return redirect()->route('login');
        } else {
            return 'user not found';
        }
    }

    public function refer(Request $request, $user){
        $af_system = Affilate::first();
        if($af_system && $af_system->enable_affilate == '1'){
            $findreferal = User::firstWhere('refer_code',$request->refer_code);
            if(!$findreferal){
                return back()->withInput()->withErrors([
                    'refercode' =>__('Refer code is invalid !')
                ]);
            }
        }
        if($af_system && $af_system->enable_affilate == '1'){
            $findreferal->getReferals()->create([
                'log' => 'Refer successfull',
                'refer_user_id' => $user->id,
                'user_id' => $findreferal->id,
                'amount' => $af_system->refer_amount,
                'procces' => $af_system->enable_purchase == 1 ? 0 : 1
            ]);
            if(!$findreferal->wallet){
                $w = $findreferal->wallet()->create([
                    'balance' => $af_system->refer_amount,
                    'status'  =>  '1',
                ]);
                $w->wallethistory()->create([
                    'type' => 'Credit',
                    'log' => 'Referal bonus',
                    'amount' => $af_system->refer_amount,
                    'txn_id' => str_random(8),
                    'expire_at' => date("Y-m-d", strtotime(date('Y-m-d').'+365 days'))
                ]);
            }
            if(isset($findreferal->wallet) && $findreferal->wallet->status == 1){
                $findreferal->wallet()->update([
                    'balance' => $findreferal->wallet->balance + $af_system->refer_amount
                ]);
                $findreferal->wallet->wallethistory()->create([
                    'type' => 'Credit',
                    'log' => 'Referal bonus',
                    'amount' => $af_system->refer_amount,
                    'txn_id' => str_random(8),
                    'expire_at' => date("Y-m-d", strtotime(date('Y-m-d').'+365 days'))
                ]);
            }
        }
    }

    public function freesubscribe($thisuser){
        $start=Carbon::now();
        $end=$start->addDays($this->config->free_days);
        $payment_id=mt_rand(10000000000000, 99999999999999);
        $created_subscription = PaypalSubscription::create([
            'user_id' => $thisuser->id,
            'payment_id' => $payment_id,
            'user_name' => $thisuser->name,
            'package_id' => 0,
            'price' => 0,
            'status' => 1,
            'method' => 'free',
            'subscription_from' => Carbon::now(),
            'subscription_to' => $end
        ]);
        $ps=PaypalSubscription::where('user_id',$thisuser->id)->first();
        $to= Str::substr($ps->subscription_to,0, 10);
        $from= Str::substr($ps->subscription_from,0, 10);
        $title=$this->config->free_days.' Days '.__('staticwords.freetrial');
        $desc=__('staticwords.freetrialtext').' '.$from.' to '.$to;
        $movie_id=NULL;
        $tvid=NULL;
        $user=$thisuser->id;
        User::find($thisuser->id)->notify(new MyNotification($title,$desc,$movie_id,$tvid,$user));
    }

    /**
     * After login process - copied from LoginController
     * Handles user screen management and subscription validation
     */
    public function afterLoginProcess($auth)
    {
        if ($auth->is_admin!=1) { 
            // Check and update screen usage status on login
            if (isset($this->button->multiplescreen) && $this->button->multiplescreen == 1) {
                $current_screen = Multiplescreen::where('user_id', $auth->id)->first();

                if (!isset($current_screen)) {
                    return redirect()->route('manageprofile', Auth::user()->id);
                }

                // Set the user's nickname
                $nickname = $auth->nickname;

                // Initialize variable to track if any screen is available
                $screenAvailable = false;

                // Iterate through each screen column to check availability
                for ($i = 1; $i <= $current_screen->package->screens; $i++) {
                    $screenNameColumn = "screen_" . $i . "_used";

                    if ($current_screen->$screenNameColumn == 'NO') {
                        // Update the usage status of the available screen
                        $current_screen->$screenNameColumn = 'NO';
                        $current_screen->save();
                        Session::put('nickname', $nickname);
                        $screenAvailable = true;
                        break; // Exit loop if an available screen is found
                    }
                }

                // If no screen is available, prevent login
                if (!$screenAvailable) {
                    Auth::logout();
                    return redirect('/')->with('deleted', __('All screens for your package are currently in use'));
                }
            }
        }

        if ($this->config->free_sub==1) {
            $ps=PaypalSubscription::where('user_id',$auth->id)->first();

            if ($auth->is_admin!=1) {          
                if (!isset($ps)) {
                    $start=Carbon::now();
                    $end=$start->addDays($this->config->free_days);
                    $payment_id=mt_rand(10000000000000, 99999999999999);
                    $subscribed = 1;
                    $created_subscription = PaypalSubscription::create([
                        'user_id' => $auth->id,
                        'payment_id' => $payment_id,
                        'user_name' => $auth->name,
                        'package_id' => 0,
                        'price' => 0,
                        'status' => 1,
                        'method' => 'free',
                        'subscription_from' => Carbon::now(),
                        'subscription_to' => $end
                    ]);
                    $ps=PaypalSubscription::where('user_id',auth()->user()->id)->first();
                    $to= Str::substr($ps->subscription_to,0, 10);
                    $from= Str::substr($ps->subscription_from,0, 10);
                    $desc=__('staticwords.freetrialtext').' '.$from.' to '.$to;
                    $title=$this->config->free_days.' Days '.__('staticwords.freetrial');
                
                    $movie_id=NULL;
                    $tvid=NULL;
                    $user=$auth->id;
                    User::findOrFail($auth->id)->notify(new MyNotification($title,$desc,$movie_id,$tvid,$user));
                }
            }
        }
        
        $subscribed = null;
        if ($auth->is_admin == 1 ||  $auth->is_assistant == 1) {
            $subscribed = 1;
            return redirect('/admin');
        }
        else if(Auth::user()->is_blocked == 1){
            Auth::logout();
            return back()->with('deleted',__('You Do Not Have Access to This Site Anymore. You are Blocked.'));
        } 
        else {
            session()->put('user',['email' => Auth::user()->email, 'name' => Auth::user()->name]);
        
            $current_date = Carbon::now()->toDateString();
            Stripe::setApiKey(env('STRIPE_SECRET'));
            if ($auth->stripe_id != null) {
                $customer = Customer::retrieve($auth->stripe_id);
            }
            $paypal = PaypalSubscription::where('user_id',$auth->id)->where('status','=',1)->where('method','!=','free')->orderBy('created_at','desc')->first(); 

            $paypal_for_free = PaypalSubscription::where('user_id',$auth->id)->where('status','=',1)->where('method','free')->orderBy('created_at','desc')->first(); 

            if (isset($customer)) {         
                $alldata = $auth->subscriptions;
                $data = $alldata->last();      
            } 
            if (isset($paypal) && $paypal != null && $paypal->count() >0) {
                $last = $paypal;
            } 
            $stripedate = isset($data) ? $data->created_at : null;
            $paydate = isset($last) ? $last->created_at : null;
            
            if($stripedate > $paydate){
                if($auth->subscribed($data->name) && date($current_date) <= date($data->subscription_to)){
                    $subscribed = 1;
                    if($data->ends_at != null){
                        return redirect('/')->with('deleted', __('Please resume your subscription!'));
                    }  
                    else{
                        if(isset($data->stripe_plan) && $data->stripe_plan != NULL){
                            $planmenus= DB::table('package_menu')->where('package_id', $data->stripe_plan)->get();
                            if(isset($planmenus)){ 
                                foreach ($planmenus as $key => $value) {
                                    $menus[]=$value->menu_id;
                                }
                            }
                            if(isset($menus)){ 
                                $nav_menus=Menu::whereIn('id',$menus)->get();
                                foreach ($nav_menus as $nav => $menus) {
                                    return redirect($menus->slug);
                                }
                            }               
                        }else{
                            if($this->config->remove_landing_page == 1){
                                return redirect('account');
                            }else{
                                return redirect('/');
                            }
                        }
                    }
                }else {
                    return redirect('/')->with('deleted', __('Your subscription has been expired!'));
                }
            }
            elseif($stripedate < $paydate){
                if (date($current_date) <= date($last->subscription_to)){
                    if($last->status == 1) {
                        $subscribed = 1;
                        if(isset($last->plan['plan_id']) && $last->plan['plan_id'] != NULL){
                            $planmenus= DB::table('package_menu')->where('package_id', $last->plan['plan_id'])->get();
                            if(isset($planmenus)){ 
                                foreach ($planmenus as $key => $value) {
                                    $menus[]=$value->menu_id;
                                }
                            }
                            if(isset($menus)){ 
                                $nav_menus=Menu::whereIn('id',$menus)->get();
                                foreach ($nav_menus as $nav => $menus) {
                                    return redirect($menus->slug);
                                }
                            } 
                        }else{
                            if($this->config->remove_landing_page == 1){
                                return redirect('account');
                            }else{
                                return redirect('/');
                            }
                        }
                    }
                    else{
                        return redirect('/')->with('deleted', __('Please resume your subscription!'));
                    }                    
                } else {
                    $last->status = 0;
                    $last->save();
                    return redirect('/')->with('deleted', __('Your subscription has been expired!'));
                }
            }
            else {
                if($this->config->free_sub==1) {
                    if(isset($ps)){
                        if($ps->method == 'free') {
                            $menu = Menu::all();
                            Session::put('nickname',Auth::user()->name);
                            if($this->config->remove_landing_page == 1){
                                return redirect()->route('home',$menu[0]->slug)->with('success',__('You have subscribe now!'));
                            }else{
                                return redirect('/')->with('success',__('You have subscribe now!'));
                            }
                        }else{
                            if($ps->subscription_to > date('Y-m-d h:i:s')){
                                if(isset($ps->package_id) && $ps->package_id != NULL){
                                    $planmenus= DB::table('package_menu')->where('package_id', $ps->package_id)->get();
                                    if(isset($planmenus)){ 
                                        foreach ($planmenus as $key => $value) {
                                            $menus[]=$value->menu_id;
                                        }
                                    }
                                    if(isset($menus)){ 
                                        $nav_menus=Menu::whereIn('id',$menus)->get();
                                        foreach ($nav_menus as $nav => $menus) {
                                            return redirect($menus->slug);
                                        }
                                    } 
                                }else{
                                    if($this->config->remove_landing_page == 1){
                                        return redirect('account');
                                    }else{
                                        return redirect('/');
                                    }
                                }
                            }else{
                                return redirect('account/purchaseplan')->with('deleted', __('You have no subscription please subscribe'));
                            }
                        }
                    }
                }
                elseif(isset($paypal_for_free)) { 
                    $menu = Menu::all();
                    Session::put('nickname',Auth::user()->name);
                    if($this->config->remove_landing_page == 1){
                        return redirect()->route('home',$menu[0]->slug)->with('success',__('You have subscribe now!'));
                    }else{
                        return redirect('/')->with('success',__('You have subscribe now!'));
                    }
                }
                else {
                    return redirect('account/purchaseplan')->with('deleted', __('You have no subscription please subscribe'));
                }
            }
        }
    }
}